---- #########################################################################
---- #                                                                       #
---- # Copyright (C) OpenTX                                                  #
-----#                                                                       #
---- # License GPLv2: http://www.gnu.org/licenses/gpl-2.0.html               #
---- #                                                                       #
---- # This program is free software; you can redistribute it and/or modify  #
---- # it under the terms of the GNU General Public License version 2 as     #
---- # published by the Free Software Foundation.                            #
---- #                                                                       #
---- # This program is distributed in the hope that it will be useful        #
---- # but WITHOUT ANY WARRANTY; without even the implied warranty of        #
---- # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
---- # GNU General Public License for more details.                          #
---- #                                                                       #
---- #########################################################################
local version = 'v2.4'
local VALUE = 0
local COMBO = 1

local edit = false
local page = 1
local current = 1
local refreshState = 0
local refreshIndex = 0
local pageOffset = 0
local pages = {}
local fields = {}
local modifications = {}
local telemetryPopTimeout

local margin = 1
local spacing = 8
local numberPerPage = 7

local b4Power
local b5Band
local b6Channel
local bAllFrq

local configFields = {
    {'Channel:', COMBO, 0x80, 1, {'CH1', 'CH2', 'CH3', 'CH4', 'CH5', 'CH6', 'CH7', 'CH8'}},
    {'Band:', COMBO, 0X80, 1, {'A', 'B', 'C', 'D', 'E', 'F'}},
    {'Power:', COMBO, 0X80, 1, {'PIT', '25mW', '200mW', '600mW'}},
    {'All Frq:', COMBO, 0X80, 1, {'YES', 'NO'}}
}

local frequency = {
    {'5865M', '5733M', '5705M', '5740M', '5658M', '5362M'},
    {'5845M', '5752M', '5685M', '5760M', '5695M', '5399M'},
    {'5825M', '5771M', '5665M', '5780M', '5732M', '5436M'},
    {'5805M', '5790M', '5645M', '5800M', '5769M', '5473M'},
    {'5785M', '5809M', '5885M', '5820M', '5806M', '5510M'},
    {'5765M', '5828M', '5905M', '5840M', '5843M', '5547M'},
    {'5745M', '5847M', '5925M', '5860M', '5880M', '5584M'},
    {'5725M', '5866M', '5945M', '5880M', '5917M', '5621M'}
}

local function telemetryRead(field)
    return sportTelemetryPush(0x13, 0x30, 0x0e10, field)
end

local function telemetryIdle(field)
    return sportTelemetryPush(0x13, 0x21, 0x0e10, field)
end

local function telemetryUnIdle(field)
    return sportTelemetryPush(0x13, 0x20, 0x0e10, field)
end

local function telemetryWrite(field, value)
    return sportTelemetryPush(0x13, 0x31, 0x0e10, field + value * 256)
end

local function refreshNext(event)
    if refreshState == 0 then
        if #modifications > 0 then
            telemetryWrite(modifications[1][1], modifications[1][2])
            modifications[1] = nil
            refreshIndex = 0
        elseif refreshIndex < #fields then
            local field = fields[refreshIndex + 1]
            if telemetryRead(field[3]) == true then
                refreshState = 1
                telemetryPopTimeout = getTime() + 200
            end
        end
    elseif refreshState == 1 then
        local physicId, primId, dataId, value = sportTelemetryPop()
        if primId == 0x32 and dataId == 0x0e10 then
            local fieldId = value % 256
            local field = fields[refreshIndex + 1]
            if fieldId == field[3] then
                value = math.floor(value / 256)
                b4Power = value % 256
                value = math.floor(value / 256)
                b5Band = value % 256
                value = math.floor(value / 256)
                b6Channel = value
                if b4Power > 9 then
                    bAllFrq = 1
                    b4Power = b4Power - 10
                else
                    bAllFrq = 0
                end
                fields[1][4] = b6Channel
                fields[2][4] = b5Band
                fields[3][4] = b4Power
                fields[4][4] = bAllFrq
            end
            refreshIndex = refreshIndex + 1
            refreshState = 0
        elseif getTime() > telemetryPopTimeout then
            refreshState = 0
        end
    end
end

-- Select the next or previous page
local function selectField(step)
    current = 1 + ((current + step - 1 + #fields) % #fields)
    if current > numberPerPage + pageOffset then
        pageOffset = current - numberPerPage
    elseif current <= pageOffset then
        pageOffset = current - 1
    end
end

local function addField(step)
    local field = fields[current]
    local min, max
    if field[2] == VALUE then
        min = field[5]
        max = field[6]
    elseif field[2] == COMBO then
        min = 0
        max = #field[5] - 1
    end
    if (step < 0 and field[4] > min) or (step > 0 and field[4] < max) then
        field[4] = field[4] + step
    end
end

local function drawScreenTitle(title, curPage, totalPages)
    if math.fmod(math.floor(getTime() / 100), 10) == 0 then
        title = version
    end
    if LCD_W == 480 then
        lcd.drawFilledRectangle(0, 0, LCD_W, 30, TITLE_BGCOLOR)
        lcd.drawText(1, 5, title, MENU_TITLE_COLOR)
        lcd.drawText(LCD_W - 40, 5, curPage .. '/' .. totalPages, MENU_TITLE_COLOR)
    else
        lcd.drawScreenTitle(title, curPage, totalPages)
    end
end

local function drawProgressBar()
    if LCD_W == 480 then
        local width = (300 * refreshIndex) / #fields
        lcd.drawRectangle(100, 10, 300, 6)
        lcd.drawFilledRectangle(102, 12, width, 2)
    else
        local width = (60 * refreshIndex) / #fields
        lcd.drawRectangle(45, 1, 60, 6)
        lcd.drawFilledRectangle(47, 3, width, 2)
    end
end

local function redrawFieldPage()
    lcd.clear()
    drawScreenTitle("GTrans", page, #pages)

    if refreshIndex < #fields then
        drawProgressBar()
    end

    for index = 1, 7, 1 do
        local field = fields[pageOffset + index]
        if field == nil then
            break
        end

        local attr = current == (pageOffset + index) and ((edit == true and BLINK or 0) + INVERS) or 0

        lcd.drawText(1, margin + spacing * index, field[1])

        if field[4] == nil then
            lcd.drawText(LCD_W, margin + spacing * index, '---', RIGHT + attr)
        else
            if field[2] == VALUE then
                lcd.drawNumber(LCD_W, margin + spacing * index, field[4], RIGHT + attr)
            elseif field[2] == COMBO then
                if field[4] >= 0 and field[4] < #(field[5]) then
                    lcd.drawText(LCD_W, margin + spacing * index, field[5][1 + field[4]], RIGHT + attr)
                end
            end
        end
    end

    lcd.drawText(1, margin + spacing * 6, 'Frequency:')
    local a1 = configFields[1][4]
    local a2 = configFields[2][4]
    local a3 = frequency[a1 + 1][a2 + 1]
    lcd.drawText(LCD_W, margin + spacing * 6, a3, RIGHT)

    lcd.drawText(1, margin + spacing * 7, 'Version:')
    lcd.drawText(LCD_W, margin + spacing * 7, version, RIGHT)
end

local function updateField(field)
    local p = fields[1][4]
    b6Channel = p
    p = fields[2][4]
    b5Band = p
    p = fields[3][4]
    b4Power = p
    b4Power = b4Power + (fields[4][4]) * 10

    local value = b6Channel * 256 * 256 + b5Band * 256 + b4Power
    modifications[#modifications + 1] = {field[3], value}
end

local function runFieldsPage(event)
    if event == EVT_EXIT_BREAK then
        telemetryUnIdle(0x80)
        return 2
    elseif event == EVT_ENTER_BREAK then
        if fields[current][4] ~= nil then
            edit = not edit
            if edit == false then
                updateField(fields[current])
            end
        end
    elseif edit then
    if event == EVT_VIRTUAL_INC or event == EVT_VIRTUAL_INC_REPT then
            addField(1)
    elseif event == EVT_VIRTUAL_DEC or event == EVT_VIRTUAL_DEC_REPT then
            addField(-1)
        end
    else
    if event == EVT_VIRTUAL_NEXT then
            selectField(1)
    elseif event == EVT_VIRTUAL_PREV then
            selectField(-1)
        end
    end
    redrawFieldPage()
    return 0
end

local function runConfigPage(event)
    fields = configFields
    local result = runFieldsPage(event)
    return result
end

local function selectPage(step)
    page = 1 + ((page - 1 + #pages + step) % #pages)
    refreshIndex = 0
    pageOffset = 0
end

-- Init
local function init()
    current, edit, refreshState, refreshIndex = 1, false, 0, 0
    if LCD_W == 480 then
        margin = 10
        spacing = 20
        numberPerPage = 12
    end

    pages = {runConfigPage}
    telemetryIdle(0x80)
end

-- Main
local function run(event)
    if event == nil then
        error('Cannot run as a model script!')
        return 2
  elseif event == EVT_VIRTUAL_NEXT_PAGE then
        selectPage(1)
  elseif event == EVT_VIRTUAL_PREV_PAGE then
        killEvents(event)
        selectPage(-1)
    end

    local result = pages[page](event)
    refreshNext()

    return result
end

return {init = init, run = run}
